﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Threading;

namespace S04_Projet
{
    public class Display
    {
        /// <summary>
        /// Liste des opérations possibles
        /// </summary>
        public enum Operation
        {
            Enlarge = 0,
            Shrink,
            Rotate90,
            Rotate180,
            Rotate270,
            GrayScaleLinear,
            GrayScaleLuminosity,
            Superposition,
            Filter,
            //CreateImage,
            Histogram,
            Dithering,
            Debug,
            Multithreading,
            Save
        }

        private static string[] menuItems = new string[]
        {
            "Agrandir l'image",
            "Rétrécir l'image",
            "Rotation à 90°",
            "Rotation à 180°",
            "Rotation à 270°",
            "Passage à une image en nuances de gris (linéairement)",
            "Passage à une image en nuances de gris (luminosité)",
            "Superposer avec une autre image",
            "Ajouter un filtre de convolution",
            "Générer les histogrammes de l'image",
            "Floyd-Steinberg dithering",
            "Activer le mode debug",
            "Activer le multithreading",
            "Sauvegarder"
        };

        public static long elapsedTime;
        public static string lastOperationMessage;
        public static string fileInfos;
        private static Stopwatch sw = new Stopwatch();
        private static Menu mainMenu = new Menu(menuItems);
        private static Menu filterMenu = new Menu(Filter.Noms);

        /// <summary>
        /// Charge l'image 
        /// </summary>
        /// <returns>Retourne l'objet contenant l'image chargée</returns>
        public static MyImage LoadImage()
        {
            string path;
            bool validPath = false;
            byte[] file = new byte[0];

            do
            {
                Console.WriteLine("Veuillez saisir le nom du fichier à traiter :");
                path = Console.ReadLine();
                try
                {
                    file = File.ReadAllBytes(path);
                }
                catch (Exception e)
                {
                    Console.ForegroundColor = ConsoleColor.Red;
                    Console.WriteLine("Une erreur est survenue : " + e.Message);
                    Console.ForegroundColor = ConsoleColor.White;
                }
                if (file.Length != 0) validPath = true;
            } while (!validPath);

            Console.Clear();

            sw.Start();
            MyImage image = new MyImage(path);
            sw.Stop();
            Console.Clear();

            elapsedTime = sw.ElapsedMilliseconds;
            lastOperationMessage = "Image chargée en " + elapsedTime + "ms";
            fileInfos = image.ToString();

            return image;
        }

        /// <summary>
        /// Demande quelles opérations l'on souhaite effectuer sur l'image
        /// </summary>
        /// <returns>Retourne le type d'opération souhaitée</returns>
        public static Operation AskForOperation()
        {
            Console.Clear();
            Console.ForegroundColor = ConsoleColor.Green;
            Console.WriteLine(lastOperationMessage);
            Console.WriteLine();
            Console.ForegroundColor = ConsoleColor.Cyan;
            Console.WriteLine(fileInfos);
            Console.ForegroundColor = ConsoleColor.White;

            int choosenItem = mainMenu.ShowMenu(0, 9);
            return (Operation)choosenItem;
        }

        public static MyImage PerformOperation(Operation ope, MyImage img)
        {
            MyImage output = null;
            Console.Clear();

            switch (ope)
            {
                case Operation.Enlarge:
                    output = EnlargeOperation(img);
                    break;
                case Operation.Shrink:
                    output = ShrinkOperation(img);
                    break;
                case Operation.Rotate90:
                    output = img.Rotate90();
                    break;
                case Operation.Rotate180:
                    output = img.Rotate180();
                    break;
                case Operation.Rotate270:
                    output = img.Rotate270();
                    break;
                case Operation.GrayScaleLinear:
                    output = GrayOperation(img, ope);
                    break;
                case Operation.GrayScaleLuminosity:
                    output = GrayOperation(img, ope);
                    break;
                case Operation.Superposition:
                    output = SuperpositionOperation(img);
                    break;
                case Operation.Filter: // TODO
                    output = FilterOperation(img);
                    break;
                case Operation.Histogram:
                    output = HistogramOperation(img);
                    break;
                case Operation.Dithering:
                    output = img.Dithering(1);
                    break;
                case Operation.Debug:
                    output = DebugOperation(img);
                    break;
                case Operation.Multithreading:
                    output = MultithreadingOperation(img);
                    break;
                case Operation.Save:
                    SaveOperation(img);
                    break;
            }
            sw.Stop();

            elapsedTime = sw.ElapsedMilliseconds;
            lastOperationMessage = String.Format("Opération {0}, effectuée avec succès en {1}ms", ope, elapsedTime);
            fileInfos = (output != null) ? output.ToString() : "";

            return output;
        }

        public static MyImage EnlargeOperation(MyImage img)
        {
            int coeff = 0;
            bool error = false;
            do
            {
                if (error) AlertMessage("Un nombre entier supérieur à 0 est attendu.");
                Console.WriteLine("Quel coefficient d'aggrandissement souhaitez-vous utiliser ?");
            } while (error = !int.TryParse(Console.ReadLine(), out coeff) || coeff <= 0);
            sw.Restart();
            return img.Enlarge(coeff);
        }

        public static MyImage ShrinkOperation(MyImage img)
        {
            int coeff = 0;
            bool error = false;
            do
            {
                if (error) AlertMessage("Un nombre entier supérieur à 0 est attendu.");
                Console.WriteLine("Quel coefficient de rétrécissement souhaitez-vous utiliser ?");
            } while (error = !int.TryParse(Console.ReadLine(), out coeff) && coeff <= 0);
            sw.Restart();
            return img.Shrink(coeff);
        }

        public static MyImage GrayOperation(MyImage img, Operation ope)
        {
            int nbScale = 0;
            bool error = false;
            do
            {
                if (error) AlertMessage("Un nombre entier supérieur à 1 et inférieur ou égal à 255 est attendu.");
                Console.WriteLine("Combien de nuances de gris souhaitez-vous obtenir ?");
            } while (error = !int.TryParse(Console.ReadLine(), out nbScale) && nbScale <= 1 && nbScale > 255);

            sw.Restart();
            if (ope == Operation.GrayScaleLinear) return img.ToGrayScale((byte)nbScale, MyImage.GrayFilterType.LINEAR);
            else return img.ToGrayScale((byte)nbScale, MyImage.GrayFilterType.LUMINOSITY);
        }

        public static MyImage SuperpositionOperation(MyImage img)
        {
            string path = "";
            MyImage secondImg = null;
            byte[] file = null;
            int x = 0;
            int y = 0;
            bool error = true;

            // Get path to image
            do
            {
                error = false;
                Console.WriteLine("Veuillez saisir le chemin de l'image que vous voulez superposer à cette image :");
                path = Console.ReadLine();
                try
                {
                    file = File.ReadAllBytes(path);
                }
                catch (Exception e)
                {
                    error = true;
                    AlertMessage("Une erreur est survenue : \n" + e.Message);
                }
            } while (error);

            secondImg = new MyImage(file);

            do
            {
                if (error) AlertMessage("Un nombre entier est attendu.");
                Console.WriteLine("A quelle coordonnée x souhaitez vous placer l'image à superposer ?");
            } while (error = !int.TryParse(Console.ReadLine(), out x));

            do
            {
                if (error) AlertMessage("Un nombre entier est attendu.");
                Console.WriteLine("A quelle coordonnée y souhaitez vous placer l'image à superposer ?");
            } while (error = !int.TryParse(Console.ReadLine(), out y));

            sw.Restart();
            return img.Superposition(secondImg, x, y);
        }

        public static MyImage FilterOperation(MyImage img)
        {
            int choix;
            MyImage newImage;
            Console.Clear();
            Console.WriteLine("Quel filtre souhaitez-vous appliquer ?");
            choix = filterMenu.ShowMenu(0,3);
            sw.Restart();
            newImage = img.Filter(Filter.Filtres[choix], Filter.Facteurs[choix]);

            return newImage;
        }

        public static MyImage HistogramOperation(MyImage img)
        {
            int width = 0;
            bool error = false;
            string path = "";

            do
            {
                if (error) AlertMessage("Un nombre entier supérieur à 0 est attendu.");
                Console.WriteLine("Quel largeur souhaitez-vous utiliser pour l'histogramme ?");
            } while (error = !int.TryParse(Console.ReadLine(), out width) || width <= 0);

            error = false;
            do
            {
                if (error) AlertMessage("Un chemin valide vers un dossier est attendu");
                Console.WriteLine("Veuillez saisir le chemin du sauvegarde des histogrammes :");
                path = Console.ReadLine();
            } while (error = !Directory.Exists(path));
            if (!path.EndsWith("\\")) path += "\\";

            sw.Restart();
            MyImage[] histograms = img.Histogram(width);
            histograms[0].Save(path + "red.bmp");
            histograms[1].Save(path + "green.bmp");
            histograms[2].Save(path + "blue.bmp");

            return img;
        }

        public static MyImage DebugOperation(MyImage img)
        {
            sw.Restart();
            if (Program.DEBUG)
            {
                Program.DEBUG = false;
                menuItems[(int)Operation.Debug] = "Activer le mode debug";
            }
            else
            {
                Program.DEBUG = true;
                menuItems[(int)Operation.Debug] = "Désactiver le mode debug";
            }
            return img;
        }

        public static MyImage MultithreadingOperation(MyImage img)
        {
            if(Program.MULTITHREADING)
            {
                Program.MULTITHREADING = false;
                menuItems[(int)Operation.Multithreading] = "Activer le multithreading";
            }
            else
            {
                Program.MULTITHREADING = true;
                menuItems[(int)Operation.Multithreading] = "Désactiver le multithreading";
            }
            return img;
        }

        public static MyImage SaveOperation(MyImage img)
        {
            string path = "";
            Console.WriteLine("Veuillez saisir le chemin de sauvegarde du fichier");
            path = Console.ReadLine();
            if (!path.EndsWith(".bmp")) path += ".bmp";

            img.Save(path);

            return null;
        }

        public unsafe static MyImage CreateImageOperation()
        {
            Options opt = new Options
            {
                offset = 54,
                fileInfoHeaderSize = 40,
                bitsPerPixel = 24,
                format = "BM"
            };
            int x, y, form, width, height;
            byte[] pixels;
            opt.height = opt.width = width = height = x = y = form = 0;
            bool error = false;

            do opt.height = AskForInteger("Longueur de l'image :", error, "Veuillez saisir un entier supérieur à 0."); while (error = opt.height <= 0);
            error = false;
            do opt.width = AskForInteger("Largeur de l'image :", error, "Veuillez saisir un entier supérieur à 0."); while (error = opt.width <= 0);
            error = false;
            do form = AskForInteger("Type de forme :\n1)Rectangle\n2)Cercle :", error, "Veuillez saisir un entier égal à 1 ou 2."); while (error = (form != 1 && form != 2));
            error = false;
            do x = AskForInteger("Coordonnée x du centre de la forme :", error, "Veuillez saisir un entier supérieur ou égal à 0 et inférieur à " + opt.width + "."); while (error = (x < 0 || x >= opt.width));
            error = false;
            do y = AskForInteger("Coordonnée y du centre de la forme :", error, "Veuillez saisir un entier supérieur ou égal à 0 et inférieur à " + opt.width + "."); while (error = (y < 0 || y >= opt.width));
            error = false;
            if (form == 1)
            {
                do height = AskForInteger("Longueur du rectangle :", error, "Veuillez saisir un entier supérieur à 0."); while (error = height <= 0);
                error = false;
                do width = AskForInteger("Largeur du rectangle :", error, "Veuillez saisir un entier supérieur à 0."); while (error = width <= 0);
                error = false;
            }
            else if (form == 2)
            {
                do width = AskForInteger("Rayon du cercle :", error, "Veuillez saisir un entier supérieur à 0."); while (error = width <= 0);
                error = false;
            }

            pixels = new byte[opt.width * opt.height];
            opt.fileSize = opt.offset + opt.width * opt.height * 3;

            sw.Restart();

            fixed (byte* pixelsPtr = pixels)
            {
                if (form == 1)
                {
                    int x0 = x - width / 2;
                    int y0 = y - height / 2;

                    for (int i = 0; i < width; i++)
                    {
                        if (x0 + i > 0 && x0 + i < opt.width)
                        {
                            if (y0 >= 0)
                                *(pixelsPtr + x0 + i + y0 * opt.width) = 255;
                            if (y0 + height < opt.height)
                                *(pixelsPtr + x0 + i + (y0 + height) * opt.width) = 255;
                        }
                    }

                    for (int i = 0; i < height; i++)
                    {
                        if (y0 + i > 0 && y0 + i < opt.height)
                        {
                            if (x0 >= 0)
                                *(pixelsPtr + x0 + (y0 + i) * opt.width) = 255;
                            if (x0 + width < opt.width)
                                *(pixelsPtr + x0 + width + (y0 + i) * opt.width) = 255;
                        }
                    }
                }
                else
                {
                    int x0;
                    int y0;
                    for (double i = 0; i < 360; i+= 0.001)
                    {
                        x0 = x + (int)(width * Math.Cos(i));
                        y0 = y + (int)(width * Math.Sin(i));

                        *(pixelsPtr + x0 + y0 * opt.width) = 255;
                    }
                }
            }

            return new MyImage(opt, pixels, pixels, pixels);
        }

        public static void AlertMessage(string msg)
        {
            Console.ForegroundColor = ConsoleColor.Red;
            Console.WriteLine(msg);
            Console.ForegroundColor = ConsoleColor.White;
        }

        public static int AskForInteger(string msg, bool error, string errorMsg)
        {
            int result = 0;
            do
            {
                if (error) AlertMessage(errorMsg);
                Console.WriteLine(msg);

            } while (error = !int.TryParse(Console.ReadLine(), out result));
            return result;
        }
    }
}
