﻿using System;
using System.Threading;

namespace S04_Projet
{
    unsafe class MultiThreadedTask
    {
        #region Attributs
        public enum Operation
        {
            Load,
            Filter,
            Save
        }

        private enum RGB
        {
            R,
            G,
            B
        }

        private byte[] file;
        private byte[] NewRMatrix, NewGMatrix, NewBMatrix;
        byte* rPixelPtr, gPixelPtr, bPixelPtr;
        byte* filePtr;
        private int nbPixel;
        private int nbProcessors;
        private Options opt;
        private short[,] filter;
        private double factor;
        private int size;
        //private double stepSize;
        private Operation ToDo;
        #endregion

        public void SetNbPixel(int nbPixel) => this.nbPixel = nbPixel;

        public void SetFile(byte[] file) => this.file = file;

        /// <summary>
        /// Constructeur pour l'opération Filter
        /// </summary>
        /// <param name="ToDo"></param>
        /// <param name="rPixelPtr"></param>
        /// <param name="gPixelPtr"></param>
        /// <param name="bPixelPtr"></param>
        /// <param name="opt"></param>
        /// <param name="filter"></param>
        /// <param name="factor"></param>
        public unsafe MultiThreadedTask(Operation ToDo, byte* rPixelPtr, byte* gPixelPtr, byte* bPixelPtr, Options opt, short[,] filter, double factor)
        {
            if (ToDo == Operation.Filter)
            {
                this.ToDo = ToDo;
                this.opt = opt;
                this.filter = filter;
                this.factor = factor;
                this.rPixelPtr = rPixelPtr;
                this.gPixelPtr = gPixelPtr;
                this.bPixelPtr = bPixelPtr;
                nbPixel = opt.width * opt.height;
                size = filter.GetLength(0);
                if (Program.MULTITHREADING)
                    nbProcessors = Math.Min(Environment.ProcessorCount, 8);
                else
                    nbProcessors = 1;
            }
            else throw new Exception("Impossible d'utiliser ce constructeur pour ce type d'opération");
        }

        /// <summary>
        /// Constructeur pour l'opération Load
        /// </summary>
        /// <param name="ToDo"></param>
        /// <param name="opt"></param>
        /// <param name="rPixelPtr"></param>
        /// <param name="gPixelPtr"></param>
        /// <param name="bPixelPtr"></param>
        /// <param name="filePtr"></param>
        public unsafe MultiThreadedTask(Operation ToDo, Options opt, byte* rPixelPtr, byte* gPixelPtr, byte* bPixelPtr, byte* filePtr)
        {
            if (ToDo == Operation.Load || ToDo == Operation.Save)
            {
                this.rPixelPtr = rPixelPtr;
                this.gPixelPtr = gPixelPtr;
                this.bPixelPtr = bPixelPtr;
                this.ToDo = ToDo;
                this.opt = opt;
                this.filePtr = filePtr;
                nbPixel = opt.width * opt.height;
                nbProcessors = Math.Min(Environment.ProcessorCount, 8);
            }
            else throw new Exception("Impossible d'utiliser ce constructeur pour ce type d'opération");
        }

        #region Load
        public unsafe void LoadImage()
        {
            Thread[] threads = new Thread[nbProcessors];

            for (int i = 0; i < nbProcessors; i++)
            {
                threads[i] = new Thread(new ParameterizedThreadStart(LoadImageTask));
                threads[i].Start(i);
            }

            for (int i = 0; i < nbProcessors; i++)
                threads[i].Join();
        }

        private unsafe void LoadImageTask(object i)
        {
            int start = nbPixel / nbProcessors * (int)i;
            int end = nbPixel / nbProcessors * ((int)i + 1);

            int x, y;


            for (int j = start; j < end; j++)
            {
                x = j % opt.width;
                y = j / opt.width;
                *(bPixelPtr + j) = *(filePtr + opt.offset + opt.padding * y + j * 3);
                *(gPixelPtr + j) = *(filePtr + opt.offset + opt.padding * y + j * 3 + 1);
                *(rPixelPtr + j) = *(filePtr + opt.offset + opt.padding * y + j * 3 + 2);
            }
        }
        #endregion

        #region Filter
        public MyImage ApplyFilter()
        {
            int nbPixel = opt.width * opt.height;
            int pixelPerThread = nbPixel / nbProcessors;
            NewRMatrix = new byte[nbPixel];
            NewGMatrix = new byte[nbPixel];
            NewBMatrix = new byte[nbPixel];

            //stepSize = (Math.E - 1) / nbProcessors;
            Thread[] threads = new Thread[nbProcessors];

            Console.WriteLine("Creating {0} thread(s)", nbProcessors);

            for (int i = 0; i < nbProcessors; i++)
            {
                threads[i] = new Thread(new ParameterizedThreadStart(ApplyFilterTask));
                //double param = 1 + i * stepSize;
                threads[i].Start(i);
            }

            for (int i = 0; i < nbProcessors; i++)
            {
                threads[i].Join();
            }

            return new MyImage(opt, NewRMatrix, NewGMatrix, NewBMatrix);
        }

        private unsafe void ApplyFilterTask(object i)
        {
            int x, y;
            int nbPixel = opt.width * opt.height;
            int start = nbPixel / nbProcessors * (int)i;
            int end = nbPixel / nbProcessors * ((int)i + 1);

            byte[,] rMatrix = new byte[size, size];
            byte[,] gMatrix = new byte[size, size];
            byte[,] bMatrix = new byte[size, size];
            byte r, g, b;

            fixed (short* filterPtr = filter)
            {
                fixed (byte* rMatrixPtr = rMatrix, gMatrixPtr = gMatrix, bMatrixPtr = bMatrix)
                {
                    for (int j = start; j < end; j++)
                    {
                        x = j % opt.width;
                        y = j / opt.width;

                        GetMatrix(x, y, size, RGB.R, rMatrixPtr);
                        GetMatrix(x, y, size, RGB.G, gMatrixPtr);
                        GetMatrix(x, y, size, RGB.B, bMatrixPtr);
                        r = ConvolutionalResult(rMatrixPtr, filterPtr, size, factor);
                        g = ConvolutionalResult(gMatrixPtr, filterPtr, size, factor);
                        b = ConvolutionalResult(bMatrixPtr, filterPtr, size, factor);

                        NewRMatrix[j] = r;
                        NewGMatrix[j] = g;
                        NewBMatrix[j] = b;
                    }
                }
            }
        }

        /// <summary>
        /// Retourne la matrice de pixels de taille size ,de milieu (x0,y0) et de couleur RGB
        /// </summary>
        /// <param name="x0">Coordonnée verticale du pixel milieu</param>
        /// <param name="y0">Coordonnée horizontale du pixel milieu</param>
        /// <param name="size">Taille de la matrice de pixel désirée</param>
        /// <param name="rgb">Couleur souhaitée</param>
        /// <returns>Matrice de pixels de taille size ,de milieu (x0,y0) et de couleur RGB</returns>
        private unsafe void GetMatrix(int x0, int y0, int size, RGB rgb, byte* matrixPtr)
        {
            for (int i = 0; i < size * size; i++)
            {
                int x = x0 + (i % size) - ((size - 1) / 2);
                int y = y0 + (i / size) - ((size - 1) / 2);

                if (x < 0) x = 0;
                else if (x >= opt.width) x = opt.width - 1;
                if (y < 0) y = 0;
                else if (y >= opt.height) y = opt.height - 1;

                if (rgb == RGB.R)
                    *(matrixPtr + i) = *(rPixelPtr + x + y * opt.width);
                else if (rgb == RGB.G)
                    *(matrixPtr + i) = *(gPixelPtr + x + y * opt.width);
                else if (rgb == RGB.B)
                    *(matrixPtr + i) = *(bPixelPtr + x + y * opt.width);
            }
        }

        /// <summary>
        /// Calcul convolutionnel
        /// </summary>
        /// <param name="matrix">Matrice</param>
        /// <param name="conv">Convolution</param>
        /// <param name="size">Taille du filtre</param>
        /// <param name="factor">Factor de normalisation</param>
        /// <returns>Résultat du calcul convolutinnel</returns>
        public unsafe byte ConvolutionalResult(byte* matrix, short* conv, int size, double factor)
        {
            short r = 0; ;

            for (int i = 0; i < size * size; i++)
                r += (short)(*(matrix + i) * *(conv + i));

            r = (short)Math.Round(r * factor);
            if (r > 255) return 255;
            else if (r < 0) return 0;
            else return (byte)r;
        }
        #endregion

        #region Save
        public unsafe void SaveImage()
        {
            Thread[] threads = new Thread[nbProcessors];

            for (int i = 0; i < nbProcessors; i++)
            {
                threads[i] = new Thread(new ParameterizedThreadStart(SaveImageTask));
                threads[i].Start(i);
            }

            for (int i = 0; i < nbProcessors; i++)
                threads[i].Join();
        }

        private unsafe void SaveImageTask(object i)
        {
            int start = nbPixel / nbProcessors * (int)i;
            int end = nbPixel / nbProcessors * ((int)i + 1);

            int x, y;


            for (int j = start; j < end; j++)
            {
                x = j % opt.width;
                y = j / opt.width;
                *(filePtr + opt.offset + opt.padding * y + j * 3) = *(bPixelPtr + j);
                *(filePtr + opt.offset + opt.padding * y + j * 3 + 1) = *(gPixelPtr + j);
                *(filePtr + opt.offset + opt.padding * y + j * 3 + 2) = *(rPixelPtr + j);
            }
        }
        #endregion
    }
}
