﻿using System;
using System.IO;
using System.Text;

namespace S04_Projet
{
    public class MyImage
    {
        #region Attributs
        private Options opt;
        byte[] rPixels, gPixels, bPixels;
        byte[] file;
        private int nbProcessors;


        public enum GrayFilterType
        {
            LINEAR,
            LUMINOSITY
        }

        public enum RGB
        {
            R,
            G,
            B
        }
        #endregion

        #region Constructeurs & Save
        /// <summary>
        /// Constructeur prenant en paramètre un chemin vers un fichier BMP 24 bits
        /// </summary>
        /// <param name="path">Chemin vers un fichier BMP</param>
        public MyImage(string path)
        {
            opt = new Options();
            file = new byte[0];
            try
            {
                file = File.ReadAllBytes(path);
            }
            catch (Exception e)
            {
                Console.WriteLine("Une erreur est survenue : " + e.Message);

            }
            finally
            {
                if (file.Length != 0)
                {
                    nbProcessors = Environment.ProcessorCount;
                    FromFileToImage();
                }
            }
        }

        /// <summary>
        /// Constructeur prenant en paramètres un ficher BMP sous forme d'octets
        /// </summary>
        /// <param name="file">Ficher BMP en octets</param>
        public MyImage(byte[] file)
        {
            this.file = file;
            FromFileToImage();
        }

        /// <summary>
        /// Instancie l'objet à partir d'une matrice de pixel et d'options
        /// </summary>
        /// <param name="opt">Options de l'image</param>
        /// <param name="Pixels">Matrice de pixels constituant l'image</param>
        public MyImage(Options opt, byte[] rPixels, byte[] gPixels, byte[] bPixels)
        {
            this.opt = opt;
            this.rPixels = rPixels;
            this.gPixels = gPixels;
            this.bPixels = bPixels;
        }

        /// <summary>
        /// Charge une image. Multithreading WIP. Dépendante de la fonction LoadImage pour le multithreading
        /// </summary>
        private unsafe void FromFileToImage()
        {
            opt = new Options();

            #region Header
            opt.format = (char)file[0] + "" + (char)file[1];
            opt.fileSize = EndianToInt(file, 2);
            opt.offset = EndianToInt(file, 10);
            #endregion
            #region File info
            opt.fileInfoHeaderSize = EndianToInt(file, 14);
            opt.width = EndianToInt(file, 18);
            opt.padding = opt.width % 4;
            opt.height = EndianToInt(file, 22);
            opt.colorPlanesNb = EndianToInt(file, 26);
            opt.bitsPerPixel = EndianToInt(file, 28);
            opt.compressionMethod = EndianToInt(file, 30);
            opt.imgSize = EndianToInt(file, 34);
            opt.horizontalRes = EndianToInt(file, 38);
            opt.VerticalRes = EndianToInt(file, 42);
            opt.nbColor = EndianToInt(file, 46);
            opt.nbImportantColor = EndianToInt(file, 50);
            #endregion
            #region Pixel array
            int pixelNumber = opt.width * opt.height;
            rPixels = new byte[pixelNumber];
            gPixels = new byte[pixelNumber];
            bPixels = new byte[pixelNumber];


            #region Mono
            if (!Program.MULTITHREADING)
            {
                int y;
                fixed (byte* filePtr = file, rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < pixelNumber; i++)
                    {
                        y = i / opt.width;

                        *(rPtr + i) = *(filePtr + i * 3 + opt.padding * y + opt.offset + 2);
                        *(gPtr + i) = *(filePtr + i * 3 + opt.padding * y + opt.offset + 1);
                        *(bPtr + i) = *(filePtr + i * 3 + opt.padding * y + opt.offset);
                    }
                }
            }
            #endregion

            #region Multithreading
            else
            {
                fixed (byte* filePtr = file, rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    MultiThreadedTask LoadImageTask = new MultiThreadedTask(MultiThreadedTask.Operation.Load, opt, rPtr, gPtr, bPtr, filePtr);
                    LoadImageTask.LoadImage();
                }
            }
            #endregion

            file = null;

            #endregion
        }

        /// <summary>
        /// Sauvegarde l'image vers le chemin passé en paramètres
        /// </summary>
        /// <param name="output">Chemin de sortie de l'image</param>
        private unsafe void FromImageToFile(string output)
        {
            long nbPixel = (opt.height * opt.width);
            long bytesNumber = nbPixel * 3 + opt.offset + opt.height * opt.padding; // Multiply by 3 because 3 bytes / pixel
            byte[] file = new byte[bytesNumber];


            #region HEADER
            // FORMAT
            ASCIIEncoding encoding = new ASCIIEncoding();
            MixArrays(file, encoding.GetBytes(opt.format), 0);
            // FILE SIZE
            MixArrays(file, IntToEndian(opt.fileSize), 2);
            // OFFSET
            MixArrays(file, IntToEndian(opt.offset), 10);
            #endregion

            #region File Info Header
            MixArrays(file, IntToEndian(opt.fileInfoHeaderSize), 14);
            MixArrays(file, IntToEndian(opt.width), 18);
            MixArrays(file, IntToEndian(opt.height), 22);
            MixArrays(file, IntToEndian(1), 26); // Number of colors planes
            MixArrays(file, IntToEndian(opt.bitsPerPixel), 28);
            MixArrays(file, IntToEndian(opt.compressionMethod), 30);
            MixArrays(file, IntToEndian(opt.fileSize - opt.offset), 34); // Image size TO CHANGE IF NO MORE BI_RGB
            MixArrays(file, IntToEndian(opt.horizontalRes), 38);
            MixArrays(file, IntToEndian(opt.VerticalRes), 42);
            MixArrays(file, IntToEndian(opt.nbColor), 46);
            MixArrays(file, IntToEndian(0), 50); // Number of important colors
            #endregion

            #region Pixel array
            if (!Program.MULTITHREADING)
            {
                int x;
                int y;
                fixed (byte* filePtr = file)
                {
                    for (int i = 0; i < nbPixel; i++)
                    {
                        x = i % opt.width;
                        y = i / opt.width;

                        *(filePtr + i * 3 + opt.padding * y + opt.offset) = bPixels[i];
                        *(filePtr + i * 3 + opt.padding * y + opt.offset + 1) = gPixels[i];
                        *(filePtr + i * 3 + opt.padding * y + opt.offset + 2) = rPixels[i];
                    }
                }
            }

            else
            {
                fixed (byte* filePtr = file, rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    MultiThreadedTask SaveImageTask = new MultiThreadedTask(MultiThreadedTask.Operation.Save, opt, rPtr, gPtr, bPtr, filePtr);
                    SaveImageTask.SaveImage();
                }
            }
            #endregion

            try
            {
                File.WriteAllBytes(output, file);
            }
            catch (Exception e)
            {
                Console.ForegroundColor = ConsoleColor.Red;
                Console.WriteLine("Une erreur est survenue lors de la sauvegarde du fichier : \n" + e.Message);
            }
        }

        /// <summary>
        /// Sauvegarde l'image
        /// </summary>
        /// <param name="output">Chemin de sortie de l'image</param>
        public void Save(string output)
        {
            FromImageToFile(output);
        }
        #endregion

        #region Image processing
        #region Rotations
        /// <summary>
        /// Tourne l'image de 90° dans le sens horaire
        /// </summary>
        /// <returns>Image tournée de 90° dans le sens horaire</returns>
        public unsafe MyImage Rotate90()
        {
            Options options = opt.Copy();
            options.width = opt.height;
            options.height = opt.width;

            int size = opt.height * opt.width;

            byte[] newRPixel = new byte[size];
            byte[] newGPixel = new byte[size];
            byte[] newBPixel = new byte[size];

            fixed (byte* newRPixePtr = newRPixel, newGPixePtr = newGPixel, newBPixePtr = newBPixel)
            {
                fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < size; i++)
                    {
                        int x = i % opt.width;
                        int y = i / opt.width;

                        *(newRPixePtr + y + (options.height - 1 - x) * options.width) = *(rPtr + i);
                        *(newGPixePtr + y + (options.height - 1 - x) * options.width) = *(gPtr + i);
                        *(newBPixePtr + y + (options.height - 1 - x) * options.width) = *(bPtr + i);
                    }
                }
            }
            return new MyImage(options, newRPixel, newGPixel, newBPixel);
        }

        /// <summary>
        /// Tourne l'image de 180° dans le sens horaire
        /// </summary>
        /// <returns>Image tournée de 180° dans le sens horaire</returns>
        public unsafe MyImage Rotate180()
        {
            Options options = opt.Copy();
            int size = opt.height * opt.width;

            byte[] newRPixel = new byte[size];
            byte[] newGPixel = new byte[size];
            byte[] newBPixel = new byte[size];

            fixed (byte* newRPixePtr = newRPixel, newGPixePtr = newGPixel, newBPixePtr = newBPixel)
            {
                fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < options.width * options.height; i++)
                    {
                        int x = i % opt.width;
                        int y = i / opt.width;

                        *(newRPixePtr - x - 1 + (options.height - y) * options.width) = *(rPtr + i);
                        *(newGPixePtr - x - 1 + (options.height - y) * options.width) = *(gPtr + i);
                        *(newBPixePtr - x - 1 + (options.height - y) * options.width) = *(bPtr + i);
                    }
                }
            }
            return new MyImage(options, newRPixel, newGPixel, newBPixel);
        }

        /// <summary>
        /// Tourne l'image de 270° dans le sens horaire
        /// </summary>
        /// <returns>Image tournée de 270° dans le sens horaire</returns>
        public unsafe MyImage Rotate270()
        {
            Options options = opt.Copy();
            options.width = opt.height;
            options.height = opt.width;
            int size = opt.height * opt.width;

            byte[] newRPixel = new byte[size];
            byte[] newGPixel = new byte[size];
            byte[] newBPixel = new byte[size];

            fixed (byte* newRPixePtr = newRPixel, newGPixePtr = newGPixel, newBPixePtr = newBPixel)
            {
                fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < options.width * options.height; i++)
                    {
                        int x = i % opt.width;
                        int y = i / opt.width;

                        *(newRPixePtr + options.width - 1 - y + x * options.width) = *(rPtr + i);
                        *(newGPixePtr + options.width - 1 - y + x * options.width) = *(gPtr + i);
                        *(newBPixePtr + options.width - 1 - y + x * options.width) = *(bPtr + i);
                        //PixelArr[options.width - 1 - y, x] = Pixels[x, y];
                    }
                }
            }

            return new MyImage(options, newRPixel, newGPixel, newBPixel);
        }
        #endregion

        /// <summary>
        /// Passe l'image en nuances de gris
        /// </summary>
        /// <param name="scale">Nombre de nuances de gris désirées (entre 2 et 255) (2 = noir et blanc)</param>
        /// <param name="type">Type de transformation. Linéaire ou en fonction de la luminosité de la couleur</param>
        /// <returns>Image en nuances de gris</returns>
        public unsafe MyImage ToGrayScale(byte scale, GrayFilterType type)
        {
            Options options = opt.Copy();
            opt.nbColor = scale;
            int size = opt.height * opt.width;

            byte[] newRPixel = new byte[size];
            byte[] newGPixel = new byte[size];
            byte[] newBPixel = new byte[size];
            fixed (byte* newRPixePtr = newRPixel, newGPixePtr = newGPixel, newBPixePtr = newBPixel)
            {
                fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < opt.width * opt.height; i++)
                    {
                        int x = i % opt.width;
                        int y = i / opt.width;
                        if (type == GrayFilterType.LINEAR)
                        {
                            *(newRPixePtr + x + y * opt.width) =
                                *(newGPixePtr + x + y * opt.width) =
                                *(newBPixePtr + x + y * opt.width) = GetGray(*(rPtr + x + y * opt.width), *(gPtr + x + y * opt.width), *(bPtr + x + y * opt.width), scale);
                        }
                        else if (type == GrayFilterType.LUMINOSITY)
                            *(newRPixePtr + x + y * opt.width) =
                                *(newGPixePtr + x + y * opt.width) =
                                *(newBPixePtr + x + y * opt.width) = GetGrayLuminosity(*(rPtr + x + y * opt.width), *(gPtr + x + y * opt.width), *(bPtr + x + y * opt.width), scale);
                    }
                }
            }
            return new MyImage(options, newRPixel, newGPixel, newBPixel);
        }

        /// <summary>
        /// Agrandit l'image par rapport au coefficient passé en paramètre
        /// </summary>
        /// <param name="coeff">Coefficient d'agrandissement de l'image</param>
        /// <returns>Image élargie</returns>
        public unsafe MyImage Enlarge(int coeff)
        {
            Options options = opt.Copy();
            options.width *= coeff;
            options.height *= coeff;

            int nbPixel = opt.width * opt.height;
            int nbNewPixel = options.width * options.height;

            byte[] newPixelR = new byte[nbNewPixel];
            byte[] newPixelG = new byte[nbNewPixel];
            byte[] newPixelB = new byte[nbNewPixel];

            int x, y, x0, y0;

            fixed (byte* newRPtr = newPixelR, newGPtr = newPixelG, newBPtr = newPixelB)
            {
                fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < nbPixel; i++)
                    {
                        x0 = i % opt.width;
                        y0 = i / opt.width;
                        for (int j = 0; j < coeff * coeff; j++)
                        {
                            x = x0 * coeff + j % coeff;
                            y = y0 * coeff + j / coeff;

                            *(newRPtr + x + y * options.width) = *(rPtr + i);
                            *(newGPtr + x + y * options.width) = *(gPtr + i);
                            *(newBPtr + x + y * options.width) = *(bPtr + i);
                        }
                    }
                }
            }

            return new MyImage(options, newPixelR, newPixelG, newPixelB);
        }

        /// <summary>
        /// Rétrécit l'image par rapport au coefficient passé en paramètres
        /// </summary>
        /// <param name="coeff">Coefficient de rétrécissement de l'image</param>
        /// <returns>Image rétrécie</returns>
        public unsafe MyImage Shrink(int coeff)
        {
            Options options = opt.Copy();
            options.width /= coeff;
            options.height /= coeff;
            options.padding = options.width % 4;
            options.imgSize = options.width * options.height * 3 + options.padding * options.height;
            options.fileSize = options.imgSize + options.fileInfoHeaderSize;

            int size = options.width * options.height;

            byte[] newRPixel = new byte[size];
            byte[] newGPixel = new byte[size];
            byte[] newBPixel = new byte[size];

            int x0, y0, x, y;
            int r, g, b;
            fixed (byte* newRPixePtr = newRPixel, newGPixePtr = newGPixel, newBPixePtr = newBPixel)
            {
                fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < size; i++)
                    {
                        x0 = i % options.width;
                        y0 = i / options.width;

                        r = g = b = 0;

                        for (int j = 0; j < coeff * coeff; j++)
                        {
                            x = x0 * coeff + j % coeff;
                            y = y0 * coeff + j / coeff;

                            r += *(rPtr + x + y * opt.width);
                            g += *(gPtr + x + y * opt.width);
                            b += *(bPtr + x + y * opt.width);
                        }

                        r /= coeff * coeff;
                        g /= coeff * coeff;
                        b /= coeff * coeff;

                        *(newRPixePtr + x0 + y0 * options.width) = (byte)r;
                        *(newGPixePtr + x0 + y0 * options.width) = (byte)g;
                        *(newBPixePtr + x0 + y0 * options.width) = (byte)b;
                    }
                }
            }

            return new MyImage(options, newRPixel, newGPixel, newBPixel);
        }

        /// <summary>
        /// Applique un filtre de convolution sur une image
        /// </summary>
        /// <param name="filter">Filtre à appliquer</param>
        /// <param name="factor">Facteur de normalisation du filtre</param>
        /// <returns>Image modifiée avec le filtre</returns>
        public unsafe MyImage Filter(short[,] filter, double factor)
        {
            fixed (byte* rPixelPtr = rPixels, gPixelPtr = gPixels, bPixelPtr = bPixels)
            {
                MultiThreadedTask multiThreadedTask = new MultiThreadedTask(MultiThreadedTask.Operation.Filter, rPixelPtr, gPixelPtr, bPixelPtr, opt, filter, factor);
                return multiThreadedTask.ApplyFilter();
            }
        }

        /// <summary>
        /// Generate the 3 histograms of the image
        /// </summary>
        /// <param name="width"></param>
        /// <returns></returns>
        public unsafe MyImage[] Histogram(int width)
        {
            MyImage[] Images = new MyImage[3];

            int[,] numberOfRGB = new int[3, 256];
            int nbPixel = opt.width * opt.height;

            Options options = new Options
            {
                width = width,
                offset = 54,
                fileInfoHeaderSize = 40,
                bitsPerPixel = 24,
                format = "BM"
            };
            options.height = (int)(options.width / 1.6180339887);
            options.fileSize = options.offset + options.width * options.height * 3;

            byte[] newRPixel = new byte[options.width * options.height];
            byte[] newGPixel = new byte[options.width * options.height];
            byte[] newBPixel = new byte[options.width * options.height];

            int pixelsPerValue = options.width / 256;

            fixed (int* numberOfRGBPtr = numberOfRGB)
            {
                fixed (byte* rPixelPtr = rPixels, gPixelPtr = gPixels, bPixelPtr = bPixels)
                {
                    for (int i = 0; i < nbPixel; i++)
                    {
                        *(numberOfRGBPtr + *(rPixelPtr + i)) += 1;
                        *(numberOfRGBPtr + 256 * 1 + *(gPixelPtr + i)) += 1;
                        *(numberOfRGBPtr + 256 * 2 + *(bPixelPtr + i)) += 1;
                    }
                }

                int maxR = Max(numberOfRGBPtr, 256);
                int maxG = Max(numberOfRGBPtr + 256, 256);
                int maxB = Max(numberOfRGBPtr + 256 * 2, 256);

                for (int i = 0; i < 256; i++)
                {
                    *(numberOfRGBPtr + i) = (*(numberOfRGBPtr + i) * options.height) / maxR;
                    *(numberOfRGBPtr + 256 * 1 + i) = (*(numberOfRGBPtr + 256 * 1 + i) * options.height) / maxG;
                    *(numberOfRGBPtr + 256 * 2 + i) = (*(numberOfRGBPtr + 256 * 2 + i) * options.height) / maxB;
                }

                fixed (byte* newRPixelPtr = newRPixel, newGPixelPtr = newGPixel, newBPixelPtr = newBPixel)
                {
                    for (int i = 0; i < 256; i++)
                    {
                        for (int j = 0; j < options.height; j++)
                        {
                            for (int k = 0; k < pixelsPerValue; k++)
                            {
                                if (*(numberOfRGBPtr + i) >= j) *(newRPixelPtr + i * pixelsPerValue + k + j * options.width) = 255;
                                if (*(numberOfRGBPtr + 256 + i) >= j) *(newGPixelPtr + i * pixelsPerValue + k + j * options.width) = 255;
                                if (*(numberOfRGBPtr + 265 * 2 + i) >= j) *(newBPixelPtr + i * pixelsPerValue + k + j * options.width) = 255;
                            }
                        }
                    }
                }
            }

            Images[0] = new MyImage(options, newRPixel, new byte[options.width * options.height], new byte[options.width * options.height]);
            Images[1] = new MyImage(options, new byte[options.width * options.height], newGPixel, new byte[options.width * options.height]);
            Images[2] = new MyImage(options, new byte[options.width * options.height], new byte[options.width * options.height], newBPixel);
            return Images;
        }

        public unsafe MyImage Superposition(MyImage img, int x, int y)
        {
            byte[] newRPixel = rPixels;
            byte[] newGPixel = gPixels;
            byte[] newBPixel = bPixels;

            fixed (byte* rPixelPtr = img.rPixels, gPixelPtr = img.gPixels, bPixelPtr = img.bPixels)
            {
                fixed (byte* newRPtr = newRPixel, newGPtr = newGPixel, newBPtr = newBPixel)
                {
                    for (int i = 0; i < img.opt.height; i++)
                    {
                        if (y + i >= opt.height) break;
                        else if (y + i >= 0)
                            for (int j = 0; j < img.opt.width; j++)
                            {
                                if (x + j >= opt.width) break;
                                else if (x + j >= 0)
                                {
                                    *(newRPtr + x + j + (y + i) * opt.width) = *(rPixelPtr + i * img.opt.width + j);
                                    *(newGPtr + x + j + (y + i) * opt.width) = *(gPixelPtr + i * img.opt.width + j);
                                    *(newBPtr + x + j + (y + i) * opt.width) = *(bPixelPtr + i * img.opt.width + j);
                                }
                            }
                    }
                }
            }

            return new MyImage(opt, newRPixel, newGPixel, newBPixel);
        }

        public unsafe MyImage Dithering(byte factor)
        {
            // cf https://en.wikipedia.org/wiki/Floyd%E2%80%93Steinberg_dithering
            int nbPixel = opt.width * opt.height;
            int rErr, gErr, bErr;
            int newR, newG, newB;
            byte[] newRPixel = new byte[nbPixel];
            byte[] newGPixel = new byte[nbPixel];
            byte[] newBPixel = new byte[nbPixel];

            for (int i = 0; i < nbPixel; i++)
            {
                newRPixel[i] = rPixels[i];
                newGPixel[i] = gPixels[i];
                newBPixel[i] = bPixels[i];
            }

            for (int y = opt.height - 1; y >= 1; y--)
            {
                for (int x = 1; x < opt.width - 1; x++)
                {
                    byte oldR = newRPixel[x + y * opt.width];
                    byte oldG = newGPixel[x + y * opt.width];
                    byte oldB = newBPixel[x + y * opt.width];

                    newR = (int)Math.Round((double)factor * oldR / 255) * (255 / factor);
                    newG = (int)Math.Round((double)factor * oldG / 255) * (255 / factor);
                    newB = (int)Math.Round((double)factor * oldB / 255) * (255 / factor);

                    newRPixel[x + y * opt.width] = (byte)newR;
                    newGPixel[x + y * opt.width] = (byte)newG;
                    newBPixel[x + y * opt.width] = (byte)newB;

                    rErr = oldR - newR;
                    gErr = oldG - newG;
                    bErr = oldB - newB;

                    newRPixel[x + 1 + y * opt.width] = ToByte(newRPixel[x + 1 + y * opt.width] + ((rErr * 7) >> 4));
                    newGPixel[x + 1 + y * opt.width] = ToByte(newGPixel[x + 1 + y * opt.width] + ((gErr * 7) >> 4));
                    newBPixel[x + 1 + y * opt.width] = ToByte(newBPixel[x + 1 + y * opt.width] + ((bErr * 7) >> 4));

                    newRPixel[x - 1 + (y - 1) * opt.width] = ToByte(newRPixel[x - 1 + (y - 1) * opt.width] + ((rErr * 3) >> 4));
                    newGPixel[x - 1 + (y - 1) * opt.width] = ToByte(newGPixel[x - 1 + (y - 1) * opt.width] + ((gErr * 3) >> 4));
                    newBPixel[x - 1 + (y - 1) * opt.width] = ToByte(newBPixel[x - 1 + (y - 1) * opt.width] + ((bErr * 3) >> 4));

                    newRPixel[x + (y - 1) * opt.width] = ToByte(newRPixel[x + (y - 1) * opt.width] + ((rErr * 5) >> 4));
                    newGPixel[x + (y - 1) * opt.width] = ToByte(newGPixel[x + (y - 1) * opt.width] + ((gErr * 5) >> 4));
                    newBPixel[x + (y - 1) * opt.width] = ToByte(newBPixel[x + (y - 1) * opt.width] + ((bErr * 5) >> 4));

                    newRPixel[x + 1 + (y - 1) * opt.width] = ToByte(newRPixel[x + 1 + (y - 1) * opt.width] + (rErr >> 4));
                    newGPixel[x + 1 + (y - 1) * opt.width] = ToByte(newGPixel[x + 1 + (y - 1) * opt.width] + (gErr >> 4));
                    newBPixel[x + 1 + (y - 1) * opt.width] = ToByte(newBPixel[x + 1 + (y - 1) * opt.width] + (bErr >> 4));
                }
            }

            return new MyImage(opt, newRPixel, newGPixel, newBPixel);
        }

        private byte ToByte(int i)
        {
            if (i > 255) return 255;
            else if (i < 0) return 0;
            else return (byte)i;
        }
        #endregion

        #region Helpers
        /// <summary>
        /// Returns HSV values of the actual image
        /// </summary>
        /// <returns></returns>
        private unsafe double[,] GetHSV()
        {
            int size = opt.width * opt.height;
            double[,] HSV = new double[3, size];

            int M, m, C;
            double _H = 0;

            fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
            {
                fixed (double* hsvPtr = HSV)
                {
                    for (int i = 0; i < size; i++)
                    {
                        M = MaxRGB(i);
                        m = MinRGB(i);
                        C = M - m;

                        //Set H value
                        if (C == 0) _H = 0;
                        else if (M == *(rPtr + i)) _H = (((*gPtr + i) - *(bPtr + i)) / C) % 6;
                        else if (M == *(gPtr + i)) _H = (*(bPtr + i) - *(rPtr + i)) / C + 2;
                        else if (M == *(bPtr + i)) _H = (*(rPtr + i) - *(gPtr + i)) / C + 4;

                        *(hsvPtr + i) = 60 * _H;

                        // Set S value
                        if (M == 0) *(hsvPtr + size + i) = 0;
                        else *(hsvPtr + size + i) = C / M;

                        // Set V value
                        *(hsvPtr + size * 2 + i) = M;

                    }
                }
            }

            return HSV;
        }

        /// <summary>
        /// Retourne l'équavalent gris d'un pixel
        /// </summary>
        /// <param name="scale">Nombre de nuances de gris</param>
        /// <returns>Pixel en échelle de gris</returns>
        private byte GetGray(byte r, byte g, byte b, byte scale)
        {
            byte total = (byte)((r + g + b) / 3);
            total = (byte)(Math.Round((double)total / (255 / (scale - 1))) * (255 / (scale - 1)));
            return total;
        }

        /// <summary>
        /// Retourne l'équivalent gris d'un pixel en prenant en compte les coefficients de luminosité
        /// des couleurs RGB
        /// </summary>
        /// <param name="scale"></param>
        /// <returns></returns>
        private byte GetGrayLuminosity(byte r, byte g, byte b, byte scale)
        {
            byte total = (byte)(0.21 * r + 0.72 * g + 0.07 * b);
            total = (byte)(Math.Round((double)total / (255 / (scale - 1))) * (255 / (scale - 1)));
            return total;
        }

        private unsafe byte MaxRGB(int i)
        {
            fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
            {
                byte max = *(rPtr + i) > (*gPtr + i) ? *(rPtr + i) : *(gPtr + i);
                max = *(bPtr + i) > max ? *(bPtr + i) : max;
                return max;
            }
        }

        private unsafe byte MinRGB(int i)
        {
            fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
            {
                byte min = *(rPtr + i) < (*gPtr + i) ? *(rPtr + i) : *(gPtr + i);
                min = *(bPtr + i) < min ? *(bPtr + i) : min;
                return min;
            }
        }

        private unsafe int Max(int* tab, int length)
        {
            int max = 0;
            for (int i = 0; i < length; i++)
            {
                if (*(tab + i) > max) max = *(tab + i);
            }
            return max;
        }

        /// <summary>
        /// Passe d'un nombre au format Endian à un nombre entier
        /// </summary>
        /// <param name="arr">Tableau d'octets contenant le nombre au format endian</param>
        /// <param name="from">Point de départ (inclu)</param>
        /// <returns>Nombre entier</returns>
        private static int EndianToInt(byte[] arr, int from)
        {
            int somme = 0;
            for (int i = from; i < from + 4; i++)
                somme += (arr[i] << ((i - from) * 8));
            return somme;
        }

        /// <summary>
        /// Passe un nombre du format entier au format endian
        /// </summary>
        /// <param name="val">Valeur de l'entier</param>
        /// <returns>Tableau d'octets représentant l'entier au format endian</returns>
        private static byte[] IntToEndian(int val)
        {
            byte[] endian = new byte[4];
            for (int i = 0; i < 4; i++)
                endian[i] = (byte)(val >> (i * 8));
            return endian;
        }

        /// <summary>
        /// Ajoute le contenu du tableau 2 au tableau 1 à partir du point de départ
        /// </summary>
        /// <param name="arr">Tableau 1</param>
        /// <param name="toAdd">Tableau 2</param>
        /// <param name="start">Point de départ</param>
        private static void MixArrays(byte[] arr, byte[] toAdd, int start)
        {
            for (int i = start; i < start + toAdd.Length; i++)
            {
                arr[i] = toAdd[i - start];
            }
        }

        /// <summary>
        /// Chaine de charactères décrivant l'objet
        /// </summary>
        /// <returns>Chaine de charactères décrivant l'objet</returns>
        public override string ToString()
        {
            string str = "";

            str += String.Format("Format : {0}\nTaille : {1} octets", opt.format, opt.fileSize) + "\n";
            str += String.Format("Offset : {0} octets", opt.offset) + "\n";
            str += String.Format("File info header size : {0} octets", opt.fileInfoHeaderSize) + "\n";
            str += String.Format("Size : {0}x{1}px", opt.width, opt.height) + "\n";
            str += String.Format("Bits per pixel : " + opt.bitsPerPixel) + "\n";

            /*Console.WriteLine("Compression method : " + opt.compressionMethod);
            Console.WriteLine("Image size : {0} octets", opt.imgSize);
            Console.WriteLine("Horizontal res : {0}\nVertical res : {1}", opt.horizontalRes, opt.VerticalRes);*/

            return str;
        }
        #endregion

        #region WIP
        /*
        public unsafe MyImage EnlargeGradient(int coeff)
        {
            Options options = opt.Copy();
            options.width *= coeff;
            options.height *= coeff;

            int nbPixel = opt.width * opt.height;
            int nbNewPixel = options.width * options.height;

            byte[] newPixelR = new byte[nbNewPixel];
            byte[] newPixelG = new byte[nbNewPixel];
            byte[] newPixelB = new byte[nbNewPixel];

            int x, y, x0, y0;

            fixed (byte* newRPtr = newPixelR, newGPtr = newPixelG, newBPtr = newPixelB)
            {
                fixed (byte* rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < nbPixel; i++)
                    {
                        x0 = i % opt.width;
                        y0 = i / opt.width;
                        for (int j = 0; j < coeff * coeff; j++)
                        {
                            x = x0 * coeff + j % coeff;
                            y = y0 * coeff + j / coeff;

                            *(newRPtr + x + y * options.width) = *(rPtr + i);
                            *(newGPtr + x + y * options.width) = *(gPtr + i);
                            *(newBPtr + x + y * options.width) = *(bPtr + i);
                        }
                    }
                }
            }

            return new MyImage(options, newPixelR, newPixelG, newPixelB);
        }
        */
        /*
        private unsafe byte[] GenerateGradient(int x, int y, int coeff, RGB rgb, byte* rPtr, byte* gPtr, byte* bPtr)
        {
            // cf https://stackoverflow.com/a/1106986

            int squareCoeff = coeff * coeff;
            byte[,] grad = new byte[squareCoeff,3];

            byte[] A_RGB, B_RGB, C_RGB, D_RGB;
            #region Define ABCD
            int xPlus = x + 1 < opt.width ? x + 1 : x;
            int yPlus = y + 1 < opt.width ? y + 1 : y;

            A_RGB = new byte[] {
                *(rPtr + x + y * opt.width),
                *(gPtr + x + y * opt.width),
                *(bPtr + x + y * opt.width)
            };

            B_RGB = new byte[] {
                *(rPtr + xPlus + y * opt.width),
                *(gPtr + xPlus + y * opt.width),
                *(bPtr + xPlus + y * opt.width)
            };


            C_RGB = new byte[] {
                *(rPtr + xPlus + yPlus * opt.width),
                *(gPtr + xPlus + yPlus * opt.width),
                *(bPtr + xPlus + yPlus * opt.width)
            };


            D_RGB = new byte[] {
                *(rPtr + x + yPlus * opt.width),
                *(gPtr + x + yPlus * opt.width),
                *(bPtr + x + yPlus * opt.width)
            };
            #endregion

            int x0, y0;
            for (int i = 1; i < squareCoeff; i++)
            {
                x0 = i % coeff;
                y0 = i / coeff;
            }
        }*/
        #endregion            
    }
}