﻿using System;
using System.IO;
using System.Text;

namespace S04_Projet
{
    public class MyImage
    {
        private Options opt;
        byte[] rPixels, gPixels, bPixels;
        byte[] file;
        private int nbProcessors;

        public enum GrayFilterType
        {
            LINEAR,
            LUMINOSITY
        }

        public enum RGB
        {
            R,
            G,
            B
        }

        public Options getOptions()
        {
            return opt;
        }

        /// <summary>
        /// Constructeur prenant en paramètre un chemin vers un fichier BMP 24 bits
        /// </summary>
        /// <param name="path">Chemin vers un fichier BMP</param>
        public MyImage(string path)
        {
            opt = new Options();
            file = new byte[0];
            try
            {
                file = File.ReadAllBytes(path);
            }
            catch (Exception e)
            {
                Console.WriteLine("Une erreur est survenue : " + e.Message);

            }
            finally
            {
                if (file.Length != 0)
                {
                    nbProcessors = Environment.ProcessorCount;
                    FromFileToImage();
                }
            }
        }

        /// <summary>
        /// Constructeur prenant en paramètres un ficher BMP sous forme d'octets
        /// </summary>
        /// <param name="file">Ficher BMP en octets</param>
        public MyImage(byte[] file)
        {
            this.file = file;
            FromFileToImage();
        }

        /// <summary>
        /// Instancie l'objet à partir d'une matrice de pixel et d'options
        /// </summary>
        /// <param name="opt">Options de l'image</param>
        /// <param name="Pixels">Matrice de pixels constituant l'image</param>
        public MyImage(Options opt, byte[] rPixels, byte[] gPixels, byte[] bPixels)
        {
            this.opt = opt;
            this.rPixels = rPixels;
            this.gPixels = gPixels;
            this.bPixels = bPixels;
        }

        /// <summary>
        /// Charge une image. Multithreading WIP. Dépendante de la fonction LoadImage pour le multithreading
        /// </summary>
        private unsafe void FromFileToImage()
        {
            opt = new Options();

            #region Header
            opt.format = (char)file[0] + "" + (char)file[1];
            opt.fileSize = EndianToInt(file, 2);
            opt.offset = EndianToInt(file, 10);
            #endregion
            #region File info
            opt.fileInfoHeaderSize = EndianToInt(file, 14);
            opt.width = EndianToInt(file, 18);
            opt.padding = opt.width % 4;
            opt.height = EndianToInt(file, 22);
            opt.colorPlanesNb = EndianToInt(file, 26);
            opt.bitsPerPixel = EndianToInt(file, 28);
            opt.compressionMethod = EndianToInt(file, 30);
            opt.imgSize = EndianToInt(file, 34);
            opt.horizontalRes = EndianToInt(file, 38);
            opt.VerticalRes = EndianToInt(file, 42);
            opt.nbColor = EndianToInt(file, 46);
            opt.nbImportantColor = EndianToInt(file, 50);
            #endregion
            #region Pixel array
            int pixelNumber = opt.width * opt.height;
            rPixels = new byte[pixelNumber];
            gPixels = new byte[pixelNumber];
            bPixels = new byte[pixelNumber];


            #region Mono
            if (!Program.MULTITHREADING_LOAD)
            {
                int y;
                fixed (byte* filePtr = file, rPtr = rPixels, gPtr = gPixels, bPtr = bPixels)
                {
                    for (int i = 0; i < pixelNumber; i++)
                    {
                        y = i / opt.width;

                        *(rPtr + i) = *(filePtr + i * 3 + opt.padding * y + opt.offset + 2);
                        *(gPtr + i) = *(filePtr + i * 3 + opt.padding * y + opt.offset + 1);
                        *(bPtr + i) = *(filePtr + i * 3 + opt.padding * y + opt.offset);
                    }
                }
            }
            #endregion

            #region Multithreading
            /*else
            {
                MultiThreadedTask LoadImageTask = new MultiThreadedTask(MultiThreadedTask.Operation.Load, file, opt);
                LoadImageTask.Start();
                Pixels = LoadImageTask.getPixelMatrix();
            }*/
            #endregion

            file = null;

            #endregion
        }

        /// <summary>
        /// Sauvegarde l'image vers le chemin passé en paramètres
        /// </summary>
        /// <param name="output">Chemin de sortie de l'image</param>
        private unsafe void FromImageToFile(string output)
        {
            int nbPixel = (opt.height * opt.width);
            int bytesNumber = nbPixel * 3 + opt.offset + opt.height * opt.padding; // Multiply by 3 because 3 bytes / pixel
            byte[] file = new byte[bytesNumber];


            #region HEADER
            // FORMAT
            ASCIIEncoding encoding = new ASCIIEncoding();
            MixArrays(file, encoding.GetBytes(opt.format), 0);
            // FILE SIZE
            MixArrays(file, IntToEndian(opt.fileSize), 2);
            // OFFSET
            MixArrays(file, IntToEndian(opt.offset), 10);
            #endregion

            #region File Info Header
            MixArrays(file, IntToEndian(opt.fileInfoHeaderSize), 14);
            MixArrays(file, IntToEndian(opt.width), 18);
            MixArrays(file, IntToEndian(opt.height), 22);
            MixArrays(file, IntToEndian(1), 26); // Number of colors planes
            MixArrays(file, IntToEndian(opt.bitsPerPixel), 28);
            MixArrays(file, IntToEndian(opt.compressionMethod), 30);
            MixArrays(file, IntToEndian(opt.fileSize - opt.offset), 34); // Image size TO CHANGE IF NO MORE BI_RGB
            MixArrays(file, IntToEndian(opt.horizontalRes), 38);
            MixArrays(file, IntToEndian(opt.VerticalRes), 42);
            MixArrays(file, IntToEndian(opt.nbColor), 46);
            MixArrays(file, IntToEndian(0), 50); // Number of important colors
            #endregion

            #region Pixel array
            int x;
            int y;
            fixed (byte* filePtr = file)
            {
                for (int i = 0; i < nbPixel; i++)
                {
                    x = i % opt.width;
                    y = i / opt.width;

                    *(filePtr + i * 3 + opt.padding * y + opt.offset) = bPixels[i];
                    *(filePtr + i * 3 + opt.padding * y + opt.offset + 1) = gPixels[i];
                    *(filePtr + i * 3 + opt.padding * y + opt.offset + 2) = rPixels[i];
                }
            }
            #endregion

            File.WriteAllBytes(output, file);
        }

        /// <summary>
        /// Sauvegarde l'image
        /// </summary>
        /// <param name="output">Chemin de sortie de l'image</param>
        public void Save(string output)
        {
            FromImageToFile(output);
        }
        /*
        #region Rotations
        /// <summary>
        /// Tourne l'image de 90° dans le sens horaire
        /// </summary>
        /// <returns>Image tournée de 90° dans le sens horaire</returns>
        public MyImage Rotate90()
        {
            Options options = opt.Copy();
            options.width = opt.height;
            options.height = opt.width;
            Pixel[,] PixelArr = new Pixel[options.width, options.height];
            for (int i = 0; i < options.width * options.height; i++)
            {
                int x = i % opt.width;
                int y = i / opt.width;
                PixelArr[y, options.height - 1 - x] = Pixels[x, y];
            }
            return new MyImage(options, PixelArr);
        }

        /// <summary>
        /// Tourne l'image de 180° dans le sens horaire
        /// </summary>
        /// <returns>Image tournée de 180° dans le sens horaire</returns>
        public MyImage Rotate180()
        {
            Options options = opt.Copy();
            Pixel[,] PixelArr = new Pixel[options.width, options.height];
            for (int i = 0; i < options.width * options.height; i++)
            {
                int x = i % opt.width;
                int y = i / opt.width;
                PixelArr[options.width - x - 1, options.height - y - 1] = Pixels[x, y];
            }
            return new MyImage(options, PixelArr);
        }

        /// <summary>
        /// Tourne l'image de 270° dans le sens horaire
        /// </summary>
        /// <returns>Image tournée de 270° dans le sens horaire</returns>
        public MyImage Rotate270()
        {
            Options options = opt.Copy();
            options.width = opt.height;
            options.height = opt.width;
            Pixel[,] PixelArr = new Pixel[options.width, options.height];
            for (int i = 0; i < options.width * options.height; i++)
            {
                int x = i % opt.width;
                int y = i / opt.width;
                PixelArr[options.width - 1 - y, x] = Pixels[x, y];
            }
            return new MyImage(options, PixelArr);
        }
        #endregion
        */
        /// <summary>
        /// Agrandit l'image par rapport au coefficient passé en paramètre. WIP
        /// </summary>
        /// <param name="coeff">Coefficient d'agrandissement de l'image</param>
        /// <returns>Image élargie</returns>
        /* public MyImage Enlarge(int coeff)
         {
             Options options = opt.Copy();
             options.width *= coeff;
             options.height *= coeff;
             Pixel[,] PixelArr = new Pixel[opt.width * 2, opt.height * 2];
             for (int i = 0; i < options.width * options.height; i++)
             {
                 int x = i % options.width;
                 int y = i % options.height;

                 byte[] fromX = Pixels[x, y].getRGB();
                 byte[] toX;
                 byte[] toY;
                 byte[] toXY;

                 if (x + 1 < options.width) toX = Pixels[x + 1, y].getRGB();
                 if (y + 1 < options.height) toY = Pixels[x, y + 1].getRGB();
                 if (y + 1 < options.height && x + 1 < options.width) toXY = Pixels[x + 1, y + 1].getRGB();


             }
             return null;
         }*/

        /// <summary>
        /// Rétrécit l'image par rapport au coefficient passé en paramètres
        /// </summary>
        /// <param name="coeff">Coefficient de rétrécissement de l'image</param>
        /// <returns>Image rétrécie</returns>
        /* public MyImage Shrink(int coeff)
         {
             Options options = opt.Copy();
             options.width /= coeff;
             options.height /= coeff;
             options.padding = options.width % 4;
             options.imgSize = options.width * options.height * 3 + options.padding * options.height;
             options.fileSize = options.imgSize + options.fileInfoHeaderSize;

             Pixel[,] PixelArr = new Pixel[options.width, options.height];

             int x0, y0, x, y;
             int[] rgb;
             byte[] _rgb;

             for (int i = 0; i < options.width * options.height; i++)
             {
                 x0 = i % options.width;
                 y0 = i / options.width;

                 rgb = new int[] { 0, 0, 0 };

                 for (int j = 0; j < coeff * coeff; j++)
                 {
                     x = x0 * coeff + j % coeff;
                     y = y0 * coeff + j / coeff;

                     _rgb = Pixels[x, y].getRGB();
                     for (int k = 0; k < 3; k++)
                         rgb[k] += _rgb[k];
                 }

                 for (int j = 0; j < 3; j++)
                     rgb[j] = (byte)(rgb[j] / (coeff * coeff));

                 PixelArr[x0, y0] = new Pixel(rgb);
             }

             return new MyImage(options, PixelArr);
         }*/

        /// <summary>
        /// Passe l'image en nuances de gris
        /// </summary>
        /// <param name="scale">Nombre de nuances de gris désirées (entre 2 et 255) (2 = noir et blanc)</param>
        /// <param name="type">Type de transformation. Linéaire ou en fonction de la luminosité de la couleur</param>
        /// <returns>Image en nuances de gris</returns>
        /*
  public MyImage ToGrayScale(byte scale, GrayFilterType type)
         {
             Options options = opt.Copy();
             opt.nbColor = scale;
             Pixel[,] PixelArr = new Pixel[opt.width, opt.height];
             for (int i = 0; i < opt.width * opt.height; i++)
             {
                 int x = i % opt.width;
                 int y = i / opt.width;
                 if (type == GrayFilterType.LINEAR)
                     PixelArr[x, y] = Pixels[x, y].getGrayScale(scale);
                 else if (type == GrayFilterType.LUMINOSITY)
                     PixelArr[x, y] = Pixels[x, y].getGrayScaleLuminosity(scale);
             }
             return new MyImage(options, PixelArr);
         }*/

        public unsafe MyImage Filter(short[,] filter, double factor)
        {
            fixed (byte* rPixelPtr = rPixels, gPixelPtr = gPixels, bPixelPtr = bPixels)
            {
                MultiThreadedTask multiThreadedTask = new MultiThreadedTask(MultiThreadedTask.Operation.Filter, rPixelPtr, gPixelPtr, bPixelPtr, opt, filter, factor);
                return multiThreadedTask.ApplyFilter();
            }
        }
        
        /// <summary>
        /// Passe d'un nombre au format Endian à un nombre entier
        /// </summary>
        /// <param name="arr">Tableau d'octets contenant le nombre au format endian</param>
        /// <param name="from">Point de départ (inclu)</param>
        /// <returns>Nombre entier</returns>
        private static int EndianToInt(byte[] arr, int from)
        {
            int somme = 0;
            for (int i = from; i < from + 4; i++)
                somme += (arr[i] << ((i - from) * 8));
            return somme;
        }

        /// <summary>
        /// Passe un nombre du format entier au format endian
        /// </summary>
        /// <param name="val">Valeur de l'entier</param>
        /// <returns>Tableau d'octets représentant l'entier au format endian</returns>
        private static byte[] IntToEndian(int val)
        {
            byte[] endian = new byte[4];
            for (int i = 0; i < 4; i++)
                endian[i] = (byte)(val >> (i * 8));
            return endian;
        }

        /// <summary>
        /// Ajoute le contenu du tableau 2 au tableau 1 à partir du point de départ
        /// </summary>
        /// <param name="arr">Tableau 1</param>
        /// <param name="toAdd">Tableau 2</param>
        /// <param name="start">Point de départ</param>
        private static void MixArrays(byte[] arr, byte[] toAdd, int start)
        {
            for (int i = start; i < start + toAdd.Length; i++)
            {
                arr[i] = toAdd[i - start];
            }
        }

        /// <summary>
        /// Chaine de charactères décrivant l'objet
        /// </summary>
        /// <returns>Chaine de charactères décrivant l'objet</returns>
        public override string ToString()
        {
            string str = "";

            str += String.Format("Format : {0}\nTaille : {1} octets", opt.format, opt.fileSize) + "\n";
            str += String.Format("Offset : {0} octets", opt.offset) + "\n";
            str += String.Format("File info header size : {0} octets", opt.fileInfoHeaderSize) + "\n";
            str += String.Format("Size : {0}x{1}px", opt.width, opt.height) + "\n";
            str += String.Format("Bits per pixel : " + opt.bitsPerPixel) + "\n";

            /*Console.WriteLine("Compression method : " + opt.compressionMethod);
            Console.WriteLine("Image size : {0} octets", opt.imgSize);
            Console.WriteLine("Horizontal res : {0}\nVertical res : {1}", opt.horizontalRes, opt.VerticalRes);*/

            return str;
        }
    }
}
