import React, { useState } from "react";

import { makeStyles } from "@material-ui/core/styles"
import Grid from "@material-ui/core/Grid";
import ExpansionPanel from '@material-ui/core/ExpansionPanel';
import ExpansionPanelSummary from '@material-ui/core/ExpansionPanelSummary';
import ExpansionPanelDetails from '@material-ui/core/ExpansionPanelDetails';
import Typography from "@material-ui/core/Typography";
import TextField from "@material-ui/core/TextField";
import Button from "@material-ui/core/Button";
import ExpandMoreIcon from '@material-ui/icons/ExpandMore';

import { Danger } from "./Buttons";

const useStyles = makeStyles(theme => ({
    heading: {
        fontSize: theme.typography.pxToRem(15),
        fontWeight: theme.typography.fontWeightRegular,
    },
}));

function Category(props) {
    const classes = useStyles();
    const { category, setCoefficient, deleteElement, setElement } = props;

    const [subCategoriesCreated, setSubCategoriesCreated] = useState(0);

    const handleChange = e => {
        if (e.target.id !== "coeff")
            setElement({
                id: category.id,
                update: {
                    ...category,
                    [e.target.id]: e.target.value
                }
            });
        else
            setCoefficient(
                {
                    id: category.id,
                    coeff: parseInt(e.target.value)
                });
    }

    const addSubCategory = () => {
        setElement({
            id: category.id,
            update: {
                ...category,
                totalCoeff: category.totalCoeff + 1,
                subCategories: [
                    ...category.subCategories,
                    {
                        id: category.id + "," + subCategoriesCreated,
                        name: "Nouvelle sous catégorie",
                        coeff: 1,
                        required: true,
                        type: "",
                        upper: "Excellent",
                        lower: "Nul"
                    }
                ]
            }
        });

        setSubCategoriesCreated(subCategoriesCreated + 1);
    }

    const deleteCategory = () => {
        if (window.confirm("Êtes vous sûr de vouloir supprimer cette catégorie ?")) {
            setCoefficient({ id: category.id, coeff: 0 })
            deleteElement(category.id);
        }
    }

    return (
        <Grid container spacing={2} justify="flex-end">
            <Grid item xs={12} md={10}>
                <ExpansionPanel>
                    <ExpansionPanelSummary
                        expandIcon={<ExpandMoreIcon />}
                        aria-controls="panel1a-content"
                        id="panel1a-header"
                    >
                        <Typography className={classes.heading}>{category.name} ({Math.round(category.coeff / props.totalCoeff * 10000) / 100}%)</Typography>
                    </ExpansionPanelSummary>
                    <ExpansionPanelDetails>
                        <Grid container spacing={2}>
                            <Grid item xs={12} md={8}>
                                <TextField
                                    id="name"
                                    label="Nom de la categorie"
                                    value={category.name}
                                    onChange={handleChange}
                                    className={classes.textField}
                                    margin="normal"
                                    variant="outlined"
                                    fullWidth
                                />
                            </Grid>
                            <Grid item xs={12} md={4}>
                                <TextField
                                    id="coeff"
                                    label="Coefficient"
                                    value={category.coeff}
                                    onChange={handleChange}
                                    type="number"
                                    InputLabelProps={{
                                        shrink: true,
                                    }}
                                    margin="normal"
                                    variant="outlined"
                                />
                            </Grid>

                            <Grid item xs={12} md={10}>
                                {props.children}
                            </Grid>

                            <Grid item xs={12} md={10}>
                                <Button
                                    size="small"
                                    variant="contained"
                                    color="primary"
                                    onClick={addSubCategory}
                                >
                                    Ajouter une sous catégorie
                            </Button>
                                <Danger
                                    size="small"
                                    variant="contained"
                                    color="primary"
                                    onClick={deleteCategory}
                                >
                                    Supprimer la catégorie
                            </Danger>
                            </Grid>
                        </Grid>
                    </ExpansionPanelDetails>
                </ExpansionPanel>
            </Grid>
        </Grid>
    )
}

const shouldUpdate = (prevProp, nextProp) => nextProp.toUpdate === nextProp.category.id;

export default React.memo(Category, shouldUpdate);
