import React, { useState, useEffect } from 'react';
import { saveAs } from "file-saver";
import './App.css';

import { makeStyles } from '@material-ui/core/styles';
import Paper from "@material-ui/core/Paper";
import TextField from "@material-ui/core/TextField";
import Divider from "@material-ui/core/Divider";
import Grid from "@material-ui/core/Grid";
import AppBar from "@material-ui/core/AppBar";
import Toolbar from '@material-ui/core/Toolbar';
import Typography from '@material-ui/core/Typography';
import Button from "@material-ui/core/Button";
import CircularProgress from "@material-ui/core/CircularProgress";
import Section from "./Section";
import Category from "./Category";
import SubCategory from "./SubCategory";

const useStyles = makeStyles(theme => ({
	root: {
		flexGrow: 1
	},
	paper: {
		padding: theme.spacing(2, 3)
	},
	menuButton: {
		marginRight: theme.spacing(2),
	},
	title: {
		flexGrow: 1,
	},
	margin: {
		marginTop: theme.spacing(3),
	},
	button: {
		margin: theme.spacing(1),
	},
	rightBot: {
		bottom: "10px",
		right: "15px",
		position: "fixed"
	}
}));

function App() {
	const classes = useStyles();

	const [state, setState] = useState({
		sections: [],
		toUpdate: "",
		name: ""
	});
	const [sectionsCreated, setSectionsCreated] = useState(0);
	const [url, setURL] = useState({ url: "", requesting: false });

	useEffect(() => {
		setURL({ ...url, url: localStorage.getItem("url") || "" })
		// eslint-disable-next-line
	}, [])

	const save = () => {
		let name = prompt("Nom du fichier");
		let blob = new Blob([JSON.stringify(state)], { type: "application/json" });
		saveAs(blob, name + ".json");
	}

	const generate = () => {
		const data = { name: state.name, sections: state.sections };
		setURL({ ...url, requesting: true });

		fetch(url.url, {
			method: "POST",
			body: JSON.stringify(data)
		})
			.then(res => {
				setURL({ ...url, requesting: false });
				alert("Questionnaire crée")
			})
			.catch(err => {
				setURL({ ...url, requesting: false });
				alert(err.message)
				console.error(err);
			});
	}

	const loadConfig = () => {
		let config;
		try {
			config = JSON.parse(state.config)
		} catch (e) {
			alert("Impossible de charger cette configuration")
			return;
		}
		setState({ ...state, ...config })
	}

	const saveURL = () => {
		localStorage.setItem("url", url.url);
	}

	const addSection = () => {
		setState({
			...state,
			sections: [
				...state.sections,
				{
					id: sectionsCreated.toString(),
					name: "Nouvelle section",
					categories: [],
					totalCoeff: 0
				}
			]
		});

		setSectionsCreated(sectionsCreated + 1);
	}

	const indexesOf = id => {
		const ids = id.split(",");
		let indexes = [];

		indexes[0] = state.sections.findIndex(s => s.id === ids[0]);
		if (ids.length >= 2)
			indexes[1] = state.sections[indexes[0]].categories.findIndex(c => c.id === [ids[0], ids[1]].join(','));
		if (ids.length >= 3)
			indexes[2] = state.sections[indexes[0]].categories[indexes[1]].subCategories.findIndex(sc => sc.id === ids.join(','));

		return indexes;
	}

	const setElement = data => {
		const { id, update } = data;
		const ids = indexesOf(id);
		console.log(id, ids);

		let sections = Object.assign(state.sections);
		if (ids.length === 1)
			sections[ids[0]] = update;
		if (ids.length === 2)
			sections[ids[0]].categories[ids[1]] = update;
		else if (ids.length === 3)
			sections[ids[0]].categories[ids[1]].subCategories[ids[2]] = update;

		setState({
			name: state.name,
			toUpdate: id,
			sections
		});
	}

	const deleteElement = id => {
		const ids = indexesOf(id);
		let sections = Object.assign(state.sections);

		if (ids.length === 1)
			sections = sections.filter(s => s.id !== id);
		if (ids.length === 2)
			sections[ids[0]].categories = sections[ids[0]].categories.filter(c => c.id !== id);
		else if (ids.length === 3)
			sections[ids[0]].categories[ids[1]].subCategories = sections[ids[0]].categories[ids[1]].subCategories.filter(sc => sc.id !== id);

		console.log(sections);

		setState({
			name: state.name,
			toUpdate: "",
			sections
		})
	}

	const setCoefficient = data => {
		const { id, coeff } = data;
		const ids = indexesOf(id);

		let totalCoeff;
		let sections = Object.assign(state.sections);
		let section = sections[ids[0]];
		let category = section.categories[ids[1]];

		if (ids.length === 2) {
			category.coeff = coeff;
			totalCoeff = section.categories.reduce((acc, val) => acc + val.coeff, 0);
			section.totalCoeff = totalCoeff;
		} else if (ids.length === 3) {
			let subCategory = category.subCategories[ids[2]];
			subCategory.coeff = coeff;
			totalCoeff = category.subCategories.reduce((acc, val) => acc + val.coeff, 0);
			category.totalCoeff = totalCoeff;
		}

		setState({
			name: state.name,
			toUpdate: id,
			sections
		});
	}

	const handleChange = e => {
		if (e.target.name !== "url")
			setState({ ...state, [e.target.name]: e.target.value });
		else
			setURL({ url: e.target.value });
	}

	return (
		<Grid container justify="center" spacing={0}>
			<Grid item xs={12}>
				<AppBar position="static">
					<Toolbar>
						<Button color="inherit" style={{ fontSize: "10px", marginLeft: 'auto' }}>
							Créé par Théophile BORNON pour<br />le Pole Universitaire Léonard de Vinci
                    </Button>
					</Toolbar>
				</AppBar>
			</Grid>
			<Grid item className={classes.margin} xs={12} md={10} lg={8}>
				<Paper className={classes.paper}>
					<Grid container spacing={2}>
						<Grid item xs={12}>
							<Typography variant="h5" component="h3">
								Générateur de questionnaires
								</Typography>

							<Divider />
						</Grid>

						<Grid item xs={12} md={6} lg={4}>
							<TextField
								id="outlined-name"
								name="name"
								label="Nom du questionnaire"
								className={classes.textField}
								value={state.name}
								onChange={handleChange}
								margin="normal"
								variant="outlined"
								fullWidth
							/>
						</Grid>

						<Grid item xs={12}>
							<Grid container spacing={2}>
								{state.sections.map(section =>
									<Grid
										item
										xs={12}
										key={section.id}>
										<Section
											section={section}
											toUpdaet={state.toUpdate}
											setCoefficient={setCoefficient}
											setElement={setElement}
											deleteElement={deleteElement}
										>
											{
												section.categories.map(category =>
													<Category
														key={category.id}
														category={category}
														totalCoeff={section.totalCoeff}
														toUpdaet={state.toUpdate}
														setCoefficient={setCoefficient}
														setElement={setElement}
														deleteElement={deleteElement}
													>
														{category.subCategories.map(subCategory =>
															<SubCategory
																key={subCategory.id}
																subCategory={subCategory}
																totalCoeff={category.totalCoeff}
																toUpdaet={state.toUpdate}
																setCoefficient={setCoefficient}
																setElement={setElement}
																deleteElement={deleteElement}
															/>)
														}
													</Category>
												)
											}
										</Section>
									</Grid>
								)}
							</Grid>
						</Grid>

						<Grid item xs={12}>
							<Button
								size="medium"
								variant="contained"
								color="primary"
								onClick={addSection}
							>
								Ajouter une section
								</Button>
							<Button
								size="medium"
								variant="contained"
								color="primary"
								onClick={save}
							>
								Sauvegarder la configuration
								</Button>
							<Button
								size="medium"
								variant="contained"
								color="primary"
								onClick={generate}
							>
								Générer le formulaire
								</Button>
							{url.requesting &&
								<CircularProgress />
							}
						</Grid>
					</Grid>
				</Paper>
			</Grid>

			<Grid item className={classes.margin} xs={12} md={10} lg={8}>
				<Paper className={classes.paper}>
					<Grid container spacing={2}>
						<Grid item xs={12}>
							<Typography variant="h5" component="h3">
								Importer une configuration existante
				</Typography>

							<Divider />
						</Grid>

						<Grid item xs={12}>
							<TextField
								label="Configuration"
								fullWidth
								multiline
								margin="normal"
								variant="outlined"
								name="config"
								value={state.config}
								onChange={handleChange}
							/>
						</Grid>

						<Grid item xs={12}>
							<Button
								size="medium"
								variant="contained"
								color="primary"
								onClick={loadConfig}
							>
								Importer la configuration
				</Button>
						</Grid>
					</Grid>
				</Paper>
			</Grid>

			<Grid item className={classes.margin} xs={12} md={10} lg={8}>
				<Paper className={classes.paper}>
					<Grid container spacing={2}>
						<Grid item xs={12}>
							<Typography variant="h5" component="h3">
								URL de l'application
							</Typography>

							<Divider />
						</Grid>

						<Grid item xs={12}>
							<TextField
								label="URL"
								fullWidth
								multiline
								margin="normal"
								variant="outlined"
								name="url"
								value={url.url}
								onChange={handleChange}
							/>
						</Grid>

						<Grid item xs={12}>
							<Button
								size="medium"
								variant="contained"
								color="primary"
								onClick={saveURL}
							>
								Sauvegarder l'URL
				</Button>
						</Grid>
					</Grid>
				</Paper>
			</Grid>
		</Grid>
	);
}

export default App;
